#ifndef _fma_myri_packet_h_
#define _fma_myri_packet_h_
/*
 * FMA Myrinet packets
 */
#include "lf_myri_packet.h"
#include "lf_topo_map.h"
#include "lf_fma_comm.h"
#include "lf_xbar32.h"

#define FMA_PACKET_TYPE 0x4005
#define FMA_PROTOCOL_VERSION 9

#define FMA_HOSTNAME_LEN 64

enum fma_packet_subtype {
  FMA_SUBTYPE_FMA_TUNNEL_START,
  FMA_SUBTYPE_FMA_TUNNEL_BODY,
  FMA_SUBTYPE_FMA_TUNNEL_ACK,
  FMA_SUBTYPE_FMS_ID_QUERY,
  FMA_SUBTYPE_FMS_ID_REPLY,
  FMA_SUBTYPE_PROXY_CLIENT_ID_REQUEST,
  FMA_SUBTYPE_PROXY_CLIENT_ID_REPLY,
  FMA_SUBTYPE_PROXY_FMA_TO_FMS,
  FMA_SUBTYPE_PROXY_FMS_TO_FMA,
  FMA_SUBTYPE_TOPO_MAP,			/* peer is sending map to FMA */
  FMA_SUBTYPE_MAP_FABRIC,		/* tells FMA to map entire fabric */
  FMA_SUBTYPE_XBAR_SCOUT,
  FMA_SUBTYPE_TAGGED_XBAR_SCOUT,
  FMA_SUBTYPE_XBAR_MAP,
  FMA_SUBTYPE_TAGGED_XBAR_MAP,
  FMA_SUBTYPE_XBAR_COMPARE,
  FMA_SUBTYPE_NEW_LAG_ID,
  FMA_SUBTYPE_DIST_DONE,
  FMA_SUBTYPE_COUNT
};

/*
 * Reasons a NIC scout might be sent
 */
enum fma_nic_scout_type {
  FMA_NST_MAP,			/* used when first mapping at startup */
  FMA_NST_VERIFY,
  FMA_NST_REMAP,		/* used after first map received */
};

struct fma_myri_packet {
  struct lf_myri_packet_hdr h;

  union {

    struct fma_proxy_client_id_request { /* FMA_PROXY_CLIENT_ID_REQUEST */
      uint8_t reply_route[MYRI_MAX_ROUTE_LEN];
      uint8_t reply_route_len_8;
    } proxy_client_id_request;

    struct fma_proxy_client_id_reply { /* subtype = FMA_PROXY_CLIENT_ID_REPLY */
      uint32_t client_id_32;
    } proxy_client_id_reply;

    /* structure defining the start of a "tunneled" message
     * (a larger, higher-level message that needs reliable delivery)
     */
    struct fma_tunnel_start {
      uint32_t msg_length_32;
      uint32_t msg_type_32;
      uint32_t msg_id_32;		/* Unique ID for entire message */
      uint8_t origin_mac_addr[6];	/* MAC address of query originator */
      uint8_t reply_route[MYRI_MAX_ROUTE_LEN];
      uint8_t reply_route_len_8;
      uint32_t frag_len_32;		/* length of this fragment */
      uint8_t data[1];			/* data of the message */
    } tunnel_start;
#define LF_MAX_TUNNEL_START_LEN (LF_MAX_MYRI_RAW_PKT_LEN -	\
	(2 * sizeof(uint16_t)					\
	 + sizeof(struct fma_tunnel_start) - 1))
#define LF_TUNNEL_START_PKT_LEN(L) (2 * sizeof(uint16_t)	\
             + sizeof(struct fma_tunnel_start) - 1 + (L))

    /* a fragment of a tunneled message */
    struct fma_tunnel_body {
      uint32_t msg_id_32;		/* message ID */
      uint32_t offset_32;		/* offset into message */
      uint32_t frag_len_32;		/* length of this fragment */
      uint8_t origin_mac_addr[6];	/* MAC address of query originator */
      uint8_t data[1];			/* data of the message */
    } tunnel_body;
#define LF_MAX_TUNNEL_FRAG_LEN (LF_MAX_MYRI_RAW_PKT_LEN -	\
	(2 * sizeof(uint16_t)					\
	 + sizeof(struct fma_tunnel_body) - 1))
#define LF_TUNNEL_BODY_PKT_LEN(L) (2 * sizeof(uint16_t)	\
             + sizeof(struct fma_tunnel_body) - 1 + (L))

    /* A message acknowledging receipt of a tunnel body fragment
     * This implicitly acknowledges all messages up through this offset
     */
    struct fma_tunnel_ack {
      uint32_t msg_id_32;		/* message ID */
      uint32_t offset_32;		/* offset into message */
    } tunnel_ack;

    /*
     * struct for getting to start of topo map
     */
    struct lf_topo_map topo_map;

    /* Message from an FMA destined for a the FMS */
    struct fma_proxy_fma_to_fms {
      struct fma_proxy_fma_to_fms_hdr {
	uint32_t proxy_client_id_32;
	char hostname[LF_STRING_LEN];

	uint8_t return_route[MYRI_MAX_ROUTE_LEN];
	uint8_t return_route_len_8;

	uint32_t msg_type_32;	/* type and length of embedded message */
	uint32_t length_32;
      } h;

      uint8_t data[1];		/* start of data */
    } proxy_fma_to_fms;

    /* message for this FMA from FMS forwarded by another FMA */
    struct fma_proxy_fms_to_fma {
      struct lf_fma_msg_header h;

      uint8_t data[1];
    } proxy_fms_to_fma;

  } u;
};

/*
 * a map request packet sent to the last known mapper
 */
struct fma_map_request {
  struct lf_myri_packet_hdr hdr;

  uint32_t map_version_32;
  lf_string_t why;
};

/*
 * A NIC has a new LAG ID
 */
struct fma_new_lag_id {
  struct lf_myri_packet_hdr hdr;

  uint32_t map_version_32;
  uint8_t sender_mac_addr[6];
  uint8_t new_lag_id[LF_LAG_ID_LEN];
};

/*
 * Take the xbar compare pkt out of the union since it is
 * important to be small.
 */
struct fma_xbar_compare_pkt {
  struct lf_myri_packet_hdr hdr;

  uint32_t map_session_id_32;
  uint32_t unres_node_id_32;
  uint32_t test_node_id_32;
  uint8_t origin_mac_addr[6];	/* MAC address of query originator */
  uint8_t origin_port_8;
  uint8_t unres_port_8;
};

/*
 * An xbar scout packet, extracted from union in order to be small
 */
struct fma_xbar_scout_pkt {
  struct lf_myri_packet_hdr hdr;

  struct lf_tagged_xbar_insert tagged_xbar_insert; /* zero before sending */

  uint32_t serial_32;		/* request/reply serial number */
  uint32_t node_index_32;	/* used by mapper to ID node being explored */
  
  uint8_t origin_max_mac_addr[6]; /* max MAC address of query originator */
};

/*
 * Opaque data blob passed along inside NIC mapper scouts
 */
struct fma_nic_map_scout_opaque_data {
  uint16_t fma_protocol_16;	/* FMA protocol ID */

  uint8_t scout_type_8;		/* enum fma_nic_scout_type */
  uint8_t level_8;		/* used in mapper election */

  uint32_t node_index_32;	/* node and port we were scouting from */
  				/* constructed with LF_TOPO_NODE_INDEX */
  
  uint32_t serial_32;		/* serial number assigned by mapper */

  uint8_t origin_max_mac_addr[6]; /* highest ordinal mac addr on sender */
};

/*
 * Opaque data blob passed along inside NIC verify scouts
 */
struct fma_nic_verify_scout_opaque_data {
  uint16_t fma_protocol_16;	/* FMA protocol ID */

  uint8_t scout_type_8;		/* enum fma_nic_scout_type */
  uint8_t level_8;		/* used in mapper election */

  uint32_t serial_32;		/* request/reply serial number */
  
  uint8_t origin_nic_8;		/* which NIC on sender originated the packet */
  uint8_t origin_port_8;	/* NIC port query originated from */
  uint8_t origin_max_mac_addr[6]; /* highest ordinal mac addr on sender */
};

/*
 * Opaque data blob passed along inside NIC scout replies
 */
struct fma_nic_reply_opaque_data {
  uint16_t fma_protocol_16;		/* which protocol we speak */
  uint16_t host_nic_id_16;		/* host NIC ID on which received */
  uint16_t fma_flags_16;		/* FMA flags */
  uint8_t pad[2];

  uint32_t map_version_32;	/* version of current map */

  uint8_t mapper_mac_addr[6]; /* MAC addr of generator of current map */
  uint8_t max_mac_addr[6];	/* if known, else copy of mac_addr */
  uint8_t level_8;		/* used in mapper election */

  uint8_t nr_lag_id[LF_LAG_ID_LEN];	/* LAG ID for link aggregation */

  uint8_t nr_hostname[FMA_HOSTNAME_LEN];
};

/*
 * message describing topo map distribution results
 * FMA_SUBTYPE_DIST_DONE
 */
struct fma_dist_done_msg {
  struct lf_myri_packet_hdr hdr;

  uint32_t dist_cnt_32;		/* total number completed */
  uint32_t dist_failed_32;	/* number of completed that failed */
  uint32_t map_version_32;	/* map version we distributed */
};

#endif /* _fma_myri_packet_h_ */
